//
//  SLSZoomableView.m
//  ZoomingScrollView
//
//  Created by Brad Larson on 2/21/2009.
//

#import "SLSZoomableView.h"


@implementation SLSZoomableView

#pragma mark -
#pragma mark Initialization and teardown

- (id)initWithFrame:(CGRect)frame 
{
    if (self = [super initWithFrame:frame]) 
	{
		contentInset = UIEdgeInsetsZero;
		sizeToCenterContentIn = CGSizeZero;
		scrollOffsetCorrection = UIEdgeInsetsZero;

		previousScale = 1.0f;
		currentScale = 1.0f;
//		self.backgroundColor = [UIColor clearColor];
		self.backgroundColor = [UIColor blueColor];
		self.text = @"Hello";
    }
    return self;
}


- (void)dealloc 
{
	[text release];
    [super dealloc];
}

#pragma mark -
#pragma mark Drawing

+ (CGColorSpaceRef)genericRGBSpace;
{ 
	static CGColorSpaceRef space = NULL; 
	if(NULL == space) 
	{ 
		space = CGColorSpaceCreateDeviceRGB();
	} 
	return space; 
} 

+ (CGColorRef)blackColor; 
{ 
	static CGColorRef black = NULL; 
	if(black == NULL) 
	{ 
		CGColorSpaceRef rgbColorspace = CGColorSpaceCreateDeviceRGB();
		
		CGFloat values[4] = {0.0, 0.0, 0.0, 1.0}; 
		black = CGColorCreate(rgbColorspace, values); 
		CGColorSpaceRelease(rgbColorspace);
	} 
	return black; 
} 

#define VIEWBORDERSIZE 6.0f
#define VIEWFONTSIZE 26.0f

- (void)sizeToFit;
{
	// Calculate the text size from UIKit additions to NSString
	UIFont *theFont = [UIFont fontWithName:@"Helvetica" size:(VIEWFONTSIZE * currentScale)];
	CGSize textSize = [text sizeWithFont:theFont];
//	CGPoint viewOrigin = self.frame.origin;
	
	// Put a spacing inset around the edges of the text
	float scaledBorderSize = currentScale * VIEWBORDERSIZE;
	CGSize drawableSize = CGSizeMake(round(textSize.width + scaledBorderSize * 2.0f), round(textSize.height + scaledBorderSize * 2.0f));
	CGSize drawableSizeWithBoundaries = CGSizeMake(drawableSize.width + contentInset.left + contentInset.right, drawableSize.height + contentInset.top + contentInset.bottom);

	CGSize frameSize = CGSizeMake(MAX(drawableSizeWithBoundaries.width + scrollOffsetCorrection.left + scrollOffsetCorrection.right, sizeToCenterContentIn.width), MAX(drawableSizeWithBoundaries.height + scrollOffsetCorrection.top + scrollOffsetCorrection.bottom, sizeToCenterContentIn.height));
		
	drawableArea = CGRectMake(MAX(round(frameSize.width / 2.0f - drawableSize.width / 2.0f - scrollOffsetCorrection.right - contentInset.right), 0.0f), MAX(round(frameSize.height / 2.0f - drawableSize.height / 2.0f - scrollOffsetCorrection.bottom - contentInset.bottom), 0.0f), drawableSize.width, drawableSize.height);
	self.frame = CGRectMake(0.0f, 0.0f, frameSize.width, frameSize.height);
	[self setNeedsDisplay];
}

- (void)drawRect:(CGRect)rect 
{
	CGRect currentFrame = self.frame;
    // Set up the drawing context
	CGContextRef context = UIGraphicsGetCurrentContext();
	
	CGColorRef strokeColor = [SLSZoomableView blackColor];
	float borderRadius = currentScale * VIEWBORDERSIZE;
	
	CGContextSetLineJoin(context, kCGLineJoinRound);
	CGContextSetStrokeColorWithColor(context, strokeColor);
	CGContextSetLineWidth(context, 1.0f);
	
	// Draw a rounded rectangle with a line width of 1 pixel as border on the view
	float radius = MIN(borderRadius, floorf(currentFrame.size.width / 2.0f));
	
	CGContextSaveGState(context);
	CGContextTranslateCTM(context, drawableArea.origin.x, drawableArea.origin.y);
	
	CGContextBeginPath(context);
	CGContextMoveToPoint(context, radius, 0.0f);
	CGContextAddArcToPoint(context, drawableArea.size.width, 0.0f, drawableArea.size.width, drawableArea.size.height, radius);
	CGContextAddArcToPoint(context, drawableArea.size.width, drawableArea.size.height, 0.0f, drawableArea.size.height, radius);
	CGContextAddArcToPoint(context, 0.0f, drawableArea.size.height, 0.0f, 0.0f, radius);
	CGContextAddArcToPoint(context, 0.0f, 0.0f, drawableArea.size.width, 0.0f, radius);
	CGContextClosePath(context);
	CGContextStrokePath(context);

	// Draw the text in the center of the view
	CGContextSetFillColorWithColor(context, strokeColor);
	CGContextSetAllowsAntialiasing(context, true);
		
	UIFont *theFont = [UIFont fontWithName:@"Helvetica" size:(VIEWFONTSIZE * currentScale)];
	[text drawAtPoint:CGPointMake(borderRadius, borderRadius) withFont:theFont];

	CGContextSetAllowsAntialiasing(context, false);

	CGContextRestoreGState(context);
}

#pragma mark -
#pragma mark Helper methods

- (CGPoint)convertedContentOffsetFromPreviousScale:(CGPoint)contentOffset;
{
	CGSize currentSize = self.frame.size;
	float currentPaddingOnSide = MAX(round(currentSize.width / 2.0f - drawableArea.size.width / 2.0f), 0.0f);
	float currentPaddingOnTop = MAX(round(currentSize.height / 2.0f - drawableArea.size.height / 2.0f), 0.0f);
	
	//	return CGPointMake(round(contentOffset.x - previousSideOffsets.width * (currentScale/previousScale - 1.0f)), round(contentOffset.y - previousSideOffsets.height * (currentScale/previousScale - 1.0f)));
	CGPoint newOffset = CGPointMake(round(contentOffset.x - (previousSideOffsets.width * currentScale/previousScale - currentPaddingOnSide * 1.0f)), round(contentOffset.y - (previousSideOffsets.height * currentScale/previousScale - currentPaddingOnTop * 1.0f)));	
	if (newOffset.x < 0.0)
	{
		newOffset.x = 0.0f;
	}
	else if ((newOffset.x + sizeToCenterContentIn.width) > currentSize.width)
	{
		newOffset.x = MAX(currentSize.width - sizeToCenterContentIn.width, 0.0f);
	}
	
	if (newOffset.y < 0.0)
	{
		newOffset.y = 0.0f;
	}
	else if ((newOffset.y + sizeToCenterContentIn.height) > currentSize.height)
	{
		newOffset.y = MAX(currentSize.height - sizeToCenterContentIn.height, 0.0f);
	}
	return newOffset;
}

#pragma mark -
#pragma mark Accessors

@synthesize previousScale, currentScale;
@synthesize contentInset;
@synthesize sizeToCenterContentIn;
@synthesize text;

- (CGRect)frame;
{
	CGRect currentFrame = [super frame];
//	NSLog(@"Querying frame");
	if (sizeToCenterContentIn.width == 0.0f)
		return CGRectMake(currentFrame.origin.x, currentFrame.origin.y, MAX(currentFrame.size.width, sizeToCenterContentIn.width), MAX(currentFrame.size.height, sizeToCenterContentIn.height));
	else
		return CGRectMake(0.0f, 0.0f, MAX(currentFrame.size.width, sizeToCenterContentIn.width), MAX(currentFrame.size.height, sizeToCenterContentIn.height));
}

- (void)setTransformWithoutScaling:(CGAffineTransform)newTransform;
{
	[super setTransform:newTransform];
}

- (void)setTransform:(CGAffineTransform)newValue;
{
#ifdef DOSHARPZOOMING
	[super setTransform:CGAffineTransformScale(newValue, 1.0f / previousScale, 1.0f / previousScale)];
#else
	[super setTransform:newValue];
#endif
}

- (void)setCurrentScale:(float)newValue
{
	if (newValue == currentScale)
		return;
	
	currentScale = newValue;
	[self sizeToFit];
}

- (void)setPreviousScale:(float)newValue
{	
	previousScale = newValue;
//	previousSideOffsets = CGSizeMake(MAX(round((self.frame.size.width - scrollOffsetCorrection.left - scrollOffsetCorrection.right) / 2.0f - drawableArea.size.width / 2.0f), 0.0f), MAX(round((self.frame.size.height - scrollOffsetCorrection.top - scrollOffsetCorrection.bottom) / 2.0f - drawableArea.size.height / 2.0f), 0.0f));	
	previousSideOffsets = CGSizeMake(drawableArea.origin.x, drawableArea.origin.y);	

}

- (void)setText:(NSString *)newValue;
{
	if (newValue == text)
		return;
	
	[text release];
	text = [newValue copy];
	[self sizeToFit];
	self.previousScale = currentScale;
}

@end
